package cz.drg.clasificator.args;

import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import cz.drg.clasificator.CZDRGClasificator;
import cz.drg.clasificator.CZDRGClasificatorService;
import cz.drg.clasificator.exception.ShutdownException;
import cz.drg.clasificator.readers.DBReader;
import cz.drg.clasificator.readers.InputReader;
import cz.drg.clasificator.setting.Settings;
import cz.drg.clasificator.writers.DBWriter;
import cz.drg.clasificator.writers.OutputWriter;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.BindException;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import static cz.drg.clasificator.util.OutputHelper.*;

/**
 *
 * @author jiras
 */
public class CZDRGServiceExecutor implements ProgramExecutor, Runnable{

    private static final int DEFAULT_PORT = 8888;
    private int serverPort = DEFAULT_PORT;
    
    public static final String DASHES = new String(new char[50]).replace("\0", "-");

    
    public CZDRGServiceExecutor() {
        Settings.loadSettings();
        registerDatabaseDrivers();
    }
    
    private void registerDatabaseDrivers(){
        
        try {
            
            List<String> databaseDrivers = Settings.getProgramSettings().getDatabaseDrivers();
            
            for (String databaseDriver : databaseDrivers) {
                //register not included database drivers
                Class.forName(databaseDriver);
            }
            
        } catch (ClassNotFoundException ex) {
            dualLog("Failed to load database driver.");
            dualLog(ex.getMessage());
            System.exit(500);
        }
    }
    
    @Override
    public void stopExecution(String message) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void execute(String[] args) {
        
        if(args.length > 1){
            try{
                serverPort = Integer.parseInt(args[1]);
                dualLog("Received port: "+serverPort);
            }
            catch(NumberFormatException ex){
                dualLog("Invalid port: '"+args[1]+"', falling back to default port "+serverPort);
            }
        }
        dualLog("Starting http handler on port "+serverPort);
        new Thread(this).start();
        
    }
    
    @Override
    public void setWriter(OutputWriter writer) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void setReader(InputReader reader) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void run() {
        
        try {
            dualLog("Creating new http service on port "+serverPort+".");
            HttpServer server = HttpServer.create(new InetSocketAddress(serverPort), 0);
            server.createContext("/", new PostRequestHandler());
            server.setExecutor(null); // creates a default executor
            server.start();
            
            dualLog("Http service started on port "+serverPort+".");
            
        } 
        catch(BindException ex){
            dualLog("Port "+serverPort+" is already in use. Closing http server instance.");
        } catch (IOException ex) {
            Logger.getLogger(CZDRGServiceExecutor.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    private static class PostRequestHandler implements HttpHandler {

        private final DBReader dbReader;
        private final DBWriter dbWriter;
        private final CZDRGClasificatorService clasificator;
        
        public PostRequestHandler() {
            dbReader = new DBReader();
            dbReader.setIoClass(Settings.getProgramSettings().getIOclass("input", "database"));
            dbWriter = new DBWriter();
            dbWriter.setIoClass(Settings.getProgramSettings().getIOclass("output", "database"));
            dbWriter.setIsService(true);
            clasificator = new CZDRGClasificatorService();
            clasificator.processPMML(dbReader);
        }

        @Override
        public void handle(HttpExchange t) throws IOException {
            
            dualLog(DASHES);
            dualLog("Received "+t.getRequestMethod()+" reguest on "+t.getLocalAddress()+t.getRequestURI());
            
            for (Map.Entry<String, List<String>> entry : t.getRequestHeaders().entrySet()) {
                dualLog("Request: "+entry.getKey()+" : "+entry.getValue());
            }
            
            dualLog("Request:");
            
            BufferedReader reader = new BufferedReader(new InputStreamReader(t.getRequestBody()));
            String caseId = reader.readLine();
            dualLog("Request: "+caseId);
            dualLog(DASHES);
            dualLog("Processing id '"+caseId+"'");
            reader.close();
            
            try {
                
                dbReader.initialize(caseId);
                clasificator.setCurentCaseId(caseId);
                clasificator.startEvaluation(dbReader, dbWriter);
                t.sendResponseHeaders(HttpURLConnection.HTTP_OK, -1);
                
            } catch (ShutdownException ex) {
                
                dualLog("Evaluation failed: "+ex.getMessage());
                t.sendResponseHeaders(HttpURLConnection.HTTP_NOT_FOUND, -1);
            }
        }
    }
}

